<!-- reset-password.php -->
<?php
include('includes/config.php');

if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['token'])) {
    $token = $_GET['token'];
    $currentTime = date("Y-m-d H:i:s");
    
    $stmt = $conn->prepare("SELECT * FROM usertable WHERE reset_token = ? AND reset_expires > ?");
    $stmt->bind_param("ss", $token, $currentTime);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        die("Invalid or expired token!");
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['password'])) {
    $token = $_POST['token'];
    $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("UPDATE usertable SET password = ?, reset_token = NULL, reset_expires = NULL WHERE reset_token = ?");
    $stmt->bind_param("ss", $password, $token);
    
    if ($stmt->execute()) {
        echo "<script>
            Swal.fire({
                icon: 'success',
                title: 'Success!',
                text: 'Password updated successfully!'
            }).then(() => window.location.href = 'index.php');
        </script>";
    } else {
        echo "<script>
            Swal.fire({
                icon: 'error',
                title: 'Error!',
                text: 'Failed to update password!'
            });
        </script>";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password</title>
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        /* Base Styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Poppins', sans-serif;
        }

        /* Background Animation */
        body {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(-45deg, #ee7752, #e73c7e, #23a6d5, #23d5ab);
            background-size: 400% 400%;
            animation: gradientBG 15s ease infinite;
        }

        @keyframes gradientBG {
            0% { background-position: 0% 50%; }
            50% { background-position: 100% 50%; }
            100% { background-position: 0% 50%; }
        }

        /* Main Container */
        .forgot-container {
            width: 100%;
            max-width: 500px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
            padding: 40px;
            margin: 20px;
            opacity: 0;
            transform: translateY(20px);
            animation: fadeInUp 0.6s cubic-bezier(0.4, 0, 0.2, 1) forwards;
        }

        @keyframes fadeInUp {
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Card Styling */
        .reset-card {
            width: 100%;
            transition: all 0.4s ease;
        }

        .reset-card:hover {
            transform: perspective(1000px) rotateX(1deg) rotateY(1deg);
        }

        /* Form Elements */
        h1 {
            text-align: center;
            margin-bottom: 30px;
            color: #333;
            font-weight: 600;
        }

        .reset-input-box {
            position: relative;
            margin: 30px 0;
        }

        .reset-input-box input {
            width: 100%;
            padding: 15px 50px 15px 20px;
            background: rgba(245, 245, 245, 0.9);
            border: 2px solid transparent;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .reset-input-box input:focus {
            background: #fff;
            border-color: #7494ec;
            box-shadow: 0 0 15px rgba(116, 148, 236, 0.2);
            outline: none;
        }

        .reset-input-box i {
            position: absolute;
            right: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
            transition: 0.3s;
        }

        .reset-input-box input:focus ~ i {
            color: #7494ec;
            transform: translateY(-50%) scale(1.1);
        }

        /* Button Styling */
        .reset-btn {
            width: 100%;
            padding: 12px;
            background: linear-gradient(45deg, #7494ec, #9b7bec);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .reset-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(116, 148, 236, 0.3);
        }

        .reset-btn:active {
            transform: translateY(0);
        }

        /* Password Strength Indicator */
        .password-strength {
            height: 4px;
            background: #eee;
            border-radius: 2px;
            margin-top: 8px;
            overflow: hidden;
            position: relative;
        }

        .strength-meter {
            height: 100%;
            width: 0;
            background: red;
            transition: width 0.3s, background 0.3s;
        }

        /* Responsive Adjustments */
        @media (max-width: 600px) {
            .forgot-container {
                padding: 30px 20px;
                margin: 15px;
            }
            
            h1 {
                font-size: 24px;
                margin-bottom: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="forgot-container">
        <div class="reset-card">
            <form method="POST">
                <h1>Create New Password</h1>
                <input type="hidden" name="token" value="<?= htmlspecialchars($_GET['token']) ?>">
                
                <div class="reset-input-box">
                    <input type="password" id="newPassword" name="password" placeholder="New Password" required
                           pattern="(?=.*\d)(?=.*[a-z])(?=.*[A-Z]).{8,}"
                           title="Must contain at least one number, one uppercase and lowercase letter, and at least 8 characters">
                    <i class='bx bxs-lock-alt'></i>
                    <div class="password-strength">
                        <div class="strength-meter" id="strengthMeter"></div>
                    </div>
                </div>
                
                <div class="reset-input-box">
                    <input type="password" id="confirmPassword" placeholder="Confirm Password" required>
                    <i class='bx bxs-lock'></i>
                </div>
                
                <button type="submit" class="reset-btn">Update Password</button>
            </form>
        </div>
    </div>

    <script>
        // Password strength indicator
        const passwordInput = document.getElementById('newPassword');
        const strengthMeter = document.getElementById('strengthMeter');
        const confirmPassword = document.getElementById('confirmPassword');

        passwordInput.addEventListener('input', function() {
            const strength = calculatePasswordStrength(this.value);
            updateStrengthMeter(strength);
        });

        function calculatePasswordStrength(password) {
            let strength = 0;
            
            // Length check
            if (password.length >= 8) strength += 1;
            if (password.length >= 12) strength += 1;
            
            // Character type checks
            if (/[A-Z]/.test(password)) strength += 1;
            if (/[0-9]/.test(password)) strength += 1;
            if (/[^A-Za-z0-9]/.test(password)) strength += 1;
            
            return Math.min(strength, 5); // Max strength of 5
        }

        function updateStrengthMeter(strength) {
            const colors = ['red', 'orange', 'yellow', 'lime', 'green'];
            const width = strength * 20;
            
            strengthMeter.style.width = `${width}%`;
            strengthMeter.style.background = colors[strength - 1] || 'red';
        }

        // Confirm password validation
        confirmPassword.addEventListener('input', function() {
            if (this.value !== passwordInput.value) {
                this.setCustomValidity("Passwords don't match");
            } else {
                this.setCustomValidity('');
            }
        });

        // Form submission handling
        document.querySelector('form').addEventListener('submit', function(e) {
            if (passwordInput.value !== confirmPassword.value) {
                e.preventDefault();
                alert('Passwords do not match!');
            }
            // You can add AJAX submission here if needed
        });
    </script>
</body>
</html>